#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <assert.h>



/* Renvoie un tableau de 0 et de 1 de longueur 8 donnant
la décomposition de n en base 2. n est supposé entre 0 et 255 */
bool* bits8(unsigned char n){
   bool* res = malloc(8*sizeof(bool));
   for (int i = 0; i < 8; ++i){
      res[i] = (n >> (7-i)) & 1;
   }
   return res;
}


/* Renvoie le nom du fichier contenu les données nécessaires
   pour dessiner le caractère c */
char* img_filename(char c){
   // verifier que c est alphanumérique
   assert(('a' <= c && c <= 'z') 
       || ('A' <= c && c <= 'Z') 
       || ('0' <= c && c <= '9'));

   // construction du nom de fichier
   char* res = malloc(20*sizeof(char));
   res[0] = '\0';
   strcat(res, "lettres/");
   if ('A' <= c && c <= 'Z'){
      res[8] = c;
      res[9] = c;
      res[10] = '\0';
   } else {
      res[8] = c;
      res[9] = '\0';
   }
   strcat(res, ".txt");
   return res;
}


/* Affiche la matrice G de n lignes et m colonnes par blocs de K colonnes.
Par exemple, si G est la matrice suivante:
AAABBBCCC
AAABBBCCC
alors print_bloc(G, 2, 9, 5) affichera:
AAABB
AAABB
BCCC
BCCC
*/
void print_bloc(char** G, int n, int m, int K){
   int J = 0; // position du bloc actuel (multiple de K)
   while(J < m){
      for (int i = 0; i < n; ++i){
         // on affiche le bloc actuel, en faisant attention à ne pas dépasser du tableau
         for (int j = 0; j < K && J+j < m; ++j){
            printf("%c", G[i][J+j]);
         }
         printf("\n");
      }
      J += K;
   }
}


/* Génère une grille de pixels 8x8 pour le caractère `forme`,
   utilisant c pour les pixels actifs */
char** grille_pixels(char forme, char pix){
   char* fn = img_filename(forme);
   FILE* f = fopen(fn, "r");
   if (f == NULL){
      perror("Fichiers pour les lettres introuvables: vérifiez que le dossier lettres/ est dans le même dossier que l'exécutable");
      exit(1);
   }

   int n; // pour lire dans le fichier
   char** res = malloc(8*sizeof(char*));
   for (int i = 0; i < 8; ++i){
      res[i] = malloc(8*sizeof(char));

      fscanf(f,"%d", &n);
      bool* bits = bits8(n);
      for (int j = 0; j < 8; ++j){
         if (bits[j]){
            res[i][j] = pix;
         } else {
            res[i][j] = ' ';
         }
      }
      free(bits);
   }

   fclose(f);
   free(fn);
   return res;
}




/* Recopie le contenu de la grille tampon dans la grille G,
   avec la case (i, j) de G contenant la case (0, 0) de tampon. 
   Par exemple si G contient
   ABCD
   EFGH
   IJKL
   et tampon contient
   ZZ
   YY
   alors après avoir appelé recopier(G, tampon, 1, 2), G contient:
   ABCD
   EFZZ
   IJYY
   */
void recopier(char** G, char** tampon, int n, int m, int i, int j){
   for (int k = 0; k < n; ++k){
      for (int l = 0; l < m; ++l){
         G[i + k][j+l] = tampon[k][l];
      }
   }
}

/* renvoie true si c est alphanumérique (une lettre ou un chiffre),
   false sinon */
bool char_valide(char c){
      return (('a' <= c && c <= 'z')  // minuscule
           || ('A' <= c && c <= 'Z')  // majuscule
           || ('0' <= c && c <= '9')  // chiffre
      );
}



/* Libère la mémoire allouée pour G une grille 2D de n lignes */
void free2D(char** G, int n){
   for (int i = 0; i < n; ++i){
      free(G[i]);
   }
   free(G);
}

/* Génère une grille 2D de taille 8 x 8n (où n est la longueur de phrase)
   représentant la phrase, où chaque caractère est représenté par une image
   de 8 par 8 pixels, chaque pixel étant le caractère pix  */
char** generer_grille(char* phrase, char pix){
   int n = strlen(phrase);

   // initialiser grille vide
   char** G = malloc(8 * sizeof(char*));
   for (int i = 0; i < 8; ++i){
      G[i] = malloc(n*8*sizeof(char));
      for (int j = 0; j < n*8; ++j){
         G[i][j] = ' ';
      }
   }

   // itérer sur chaque caractère et tamponner l'image correspondante
   // sur la grille
   for (int i = 0; phrase[i] != '\0'; ++i){
      char forme = phrase[i];
      if (char_valide(forme)){
         char** tampon = grille_pixels(forme, pix);
         recopier(G, tampon, 8, 8, 0, 8*i);
         free2D(tampon, 8);         
      }
   }

   return G;
}




int main(int argc, char **argv){
   if (argc < 3){
      printf("Usage: %s [caractère] [lettres par ligne]\n", argv[0]);
      exit(1);
   }
   int K = atoi(argv[2]);
   char pix = argv[1][0];

   char* phrase = NULL;
   size_t size = 0;
   printf("Rentrez une phrase: ");
   int n = getline(&phrase, &size, stdin) - 1;
   phrase[n] = '\0'; // remplace le retour à la ligne

   char** G = generer_grille(phrase, pix);
   print_bloc(G, 8, 8*n, 8*K);

   free2D(G, 8);
   free(phrase);
   return 0;
}