#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <assert.h>
#include <string.h>

// élément de menu
struct menu_elem {
	char nom[50];
	float prix_unite;
	bool vegan;
};
typedef struct menu_elem menu_elem_t;



#define MAX_ELEMS 128
struct menu {
	int nb_elems; // Invariant: 0 <= nb_elems < MAX_ELEMS
	menu_elem_t elements[MAX_ELEMS];
};
typedef struct menu menu_t;

/* Affiche les informations de e */
void affiche_elem(menu_elem_t* e){
	printf("%s", e->nom);
	if (e->vegan){
		printf(" (V)");
	}
	printf(": %.2f€\n", e->prix_unite);
}

/* Affiche la liste des plats de m */
void affiche_menu(menu_t* m){
	printf("Menu:\n");
	for (int i = 0; i < m->nb_elems; ++i){
		printf("%d. ", i+1);
		affiche_elem(&m->elements[i]);
	}
}


/* Initialise les nb_elems premières cases de commande à 0 */
void init_command(int* commande, int nb_elems){
	for (int i = 0; i < nb_elems; ++i){
		commande[i] = 0;
	}
}

/* Lit dans le terminal les plats commandés. Pour chaque S <= i < nb_elems,
   écrit dans commande[i] le nombre d'exemplaires du i-ème plat. */
void prendre_commande(int* commande, int nb_elems){
	int plat_demande = 0;
	while(scanf("%d", &plat_demande) == 1){
		assert(1 <= plat_demande && plat_demande <= nb_elems);
		commande[plat_demande-1]++;
	}
}

/* Affiche un résumé de la commande `commande` concernant les
   plats du menu m */
void resume_commande(int* commande, menu_t* m){
	printf("Vous avez commandé:\n");
	for (int i = 0; i < m->nb_elems; ++i){
		if (commande[i] > 0){
			printf("%dx %s\n", commande[i], m->elements[i].nom);
		}
	}
}


/* Renvoie le prix total des plats commandés dans `commande` pour le
   menu m*/
float total_commande(int* commande, menu_t* m){
	float res = 0;
	for (int i = 0; i < m->nb_elems; ++i){
		res += commande[i] * m->elements[i].prix_unite;
	}
	return res;
}

/* Ajoute à m l'élément décrit par nom, prix_unite et vegan */
void ajouter_element(menu_t* m, char* nom, float prix_unite, bool vegan){
	menu_elem_t* e = &m->elements[m->nb_elems];
	e->prix_unite = prix_unite;
	e->vegan = vegan;
	strcpy(e->nom, nom);
	m->nb_elems++;
}

/* Lit dans f les informations d'un élément de menu, et les stocke dans e.
   Renvoie true si la lecture a échoué en atteignant la fin du fichier. */
bool lire_element(FILE* f, menu_elem_t* e){
	int nb_lus = fscanf(f, "%f", &e->prix_unite);
	if (nb_lus == EOF){
		return true;
	}
	int x;
	fscanf(f, "%d", &x);
	e->vegan = (x == 1);

	// jeter l'espace suivant le booléen vegan
	char poubelle;
	fscanf(f, "%c", &poubelle);
	// récupérer le reste de la ligne
	int i = 0; // nb de caractères lus
	while(fscanf(f, "%c", &e->nom[i]) != EOF && e->nom[i] != '\n'){
		i++;
	}	
	// sortie: il faut mettre un '\0' à la fin du nom.
	// si on a atteint EOF, il va dans la case i, sinon
	// il va dans la case i-1 pour remplacer le '\n' lu à la fin
	if (e->nom[i-1] == '\n'){
		e->nom[i-1] = '\0';
	} else {
		e->nom[i] = '\0';
	}
	return false;
}

void lire_menu(char* nom_fichier, menu_t* m){
	FILE* f = fopen(nom_fichier, "r");
	assert(f != NULL);
	int i = 0; // nb d'éléments lus
	while(!lire_element(f, &m->elements[i])){
		i++;
	}
	m->nb_elems = i;
	fclose(f);
}


void test(){
	menu_elem_t oeuf = {
		.nom = "Oeuf de girafe",
		.prix_unite = 8.50,
		.vegan = false
	};
	printf("Prix de l'oeuf: %f\n", oeuf.prix_unite);

	menu_t menu_3 = {
		.nb_elems = 3
	};

	strcpy(menu_3.elements[0].nom, "Pâtes au saumon");
	menu_3.elements[0].prix_unite = 11.99;
	menu_3.elements[0].vegan = false;

	strcpy(menu_3.elements[1].nom, "Curry de butternut");
	menu_3.elements[1].prix_unite = 8.99;
	menu_3.elements[1].vegan = true;

	strcpy(menu_3.elements[2].nom, "Soupe de poulet");
	menu_3.elements[2].prix_unite = 15.99;
	menu_3.elements[2].vegan = false;

	affiche_menu(&menu_3);

	ajouter_element(&menu_3, "Salade d'olives", 3.5, true);
	printf("Après ajout d'un élément:\n");
	affiche_menu(&menu_3);

	/* Test lecture de menu */
	menu_t menu_lu;
	lire_menu("menu_test.txt", &menu_lu);
	printf("Après lecture:\n");
	affiche_menu(&menu_lu);
	assert(menu_lu.nb_elems = 3);
	assert(strcmp(menu_lu.elements[1].nom, "Carte graphique grillée") == 0);
	assert(strcmp(menu_lu.elements[2].nom, "Bubble tea") == 0);
	assert(menu_lu.elements[2].vegan);
	assert(!menu_lu.elements[0].vegan);


	printf("FIN DES TESTS\n");
}


int main(int argc, char** argv){
	//test();
	if (argc < 2){
		printf("Veuillez préciser le fichier du menu\n");
		exit(1);
	}
	char* fn = argv[1];
	menu_t m;
	lire_menu(fn, &m);
	affiche_menu(&m);
	int commande[MAX_ELEMS];
	init_command(commande, m.nb_elems);
	prendre_commande(commande, m.nb_elems);
	resume_commande(commande, &m);
	printf("Total: %.2f€\n", total_commande(commande, &m));
	return 0;
}