#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <assert.h>
#include <time.h>

struct maillon{
	struct maillon* suivant;
	int val;
};
typedef struct maillon maillon_t;

struct liste {
	unsigned int taille;
	maillon_t* tete; // premier maillon
};
typedef struct liste liste_t;


/* Renvoie une liste de n entiers aléatoires entre 0 et 99 */
liste_t* liste_aleatoire(int n){
	assert(n >= 0);
	liste_t* res = malloc(sizeof(liste_t));
	res->taille = n;
	// si n = 0, on renvoie une liste vide
	if (n == 0){
		res->tete = NULL;
		return res;
	}
	// sinon, on crée un premier maillon (la tête),
	// et on ajoute les n-1 maillons suivants à la suite
	res->tete = malloc(sizeof(maillon_t));
	
	// courant représente le dernier maillon ajouté
	maillon_t* courant = res->tete; 
	courant->val = rand()%100;
	for (int i = 1; i < n; ++i){
		// créer un nouveau maillon et s'y positionner,
		// puis rentrer la valeur
		courant->suivant = malloc(sizeof(maillon_t));
		courant = courant->suivant;
		courant->val = rand()%100;
	}
	// marquer la fin de la liste avec NULL 
	courant->suivant = NULL;
	return res;
}

/* Affiche les valeurs de l dans l'ordre */
void liste_print(liste_t* l){
	assert(l != NULL);
	// parcourir la liste depuis la tête, en s'arrêtant
	// lorsque l'on atteint le NULL marquant la fin de liste
	maillon_t* m = l->tete;
	while (m != NULL){
		printf("%d ", m->val);
		m = m->suivant;
	} 
	printf("\n");
}

/*  Renvoie une liste contenant 0, 1, ..., n-1*/
liste_t* liste_range(int n){
	assert(n >= 0);
	liste_t* res = malloc(sizeof(liste_t));
	res->taille = n;
	if (n == 0){
		res->tete = NULL;
		return res;
	}

	res->tete = malloc(sizeof(maillon_t));
	
	maillon_t* courant = res->tete; 
	courant->val = 0;
	for (int i = 1; i < n; ++i){
		courant->suivant = malloc(sizeof(maillon_t));
		courant = courant->suivant;
		courant->val = i;
	}
	courant->suivant = NULL;
	return res;
}


/* Libère la liste l  */
liste_t* liste_free(liste_t* l){
	maillon_t* m = l->tete;
	while (m != NULL){
		// libérer m et passer au maillon suivant
		maillon_t* p = m; 
		m = m->suivant;
		free(p);
	}
	free(l);
} 

/* Renvoie true si l contient x, false sinon */
bool liste_recherche(int x, liste_t* l){
	for(maillon_t* m = l->tete; m != NULL; m = m->suivant){
		if (m->val == x){
			return true;
		}
	}
	return false;
}

/* Ajoute x au début de l */
void ajouter_debut(liste_t* l, int x){
	maillon_t* ancienne_tete = l->tete;

	maillon_t* nouvelle_tete = malloc(sizeof(maillon_t));
	nouvelle_tete->val = x;
	nouvelle_tete->suivant = ancienne_tete;

	l->tete = nouvelle_tete;
}

/* Ajoute x à la fin de l */
void ajouter_fin(liste_t* l, int x){
	// liste vide
	if (l->tete == NULL){
		l->tete = malloc(sizeof(maillon_t));
		l->tete->val = x;
		l->tete->suivant = NULL;
		return;
	}
	// liste non vide: la tete existe
	maillon_t* m =l->tete;
	while(m->suivant != NULL){
		m = m->suivant;
	}
	// en sortie: m est la queue actuelle
	m->suivant = malloc(sizeof(maillon_t));
	m->suivant->val = x;
	m->suivant->suivant = NULL;
}

/* Supprime la première occurrence de x dans l */
void supprimer_valeur(liste_t* l, int x){
	// Trouver le premier maillon contenant x
	maillon_t* m = l->tete;
	maillon_t* p = NULL; 
	// Invariant: p est le maillon précédant m
	while (m != NULL && m->val != x){
		p = m;
		m = m->suivant;
	}
	if (m == NULL){
		// x pas trouvé
		return;
	}

	if (p == NULL){
		// m est la tête
		l->tete = m->suivant;
		free(m);
	} else {
		// p existe
		p->suivant = m->suivant;
		free(m);
	}
}

int main(){
	srand(time(NULL));
	printf("Liste aléatoire:\n");
	liste_t* L = liste_aleatoire(10);
	liste_print(L);
	printf("\n");
	liste_free(L);

	printf("Range 10:\n");
	liste_t* L_10 = liste_range(10);
	liste_print(L_10);
	printf("\n");

	assert(liste_recherche(5, L_10));
	assert(liste_recherche(0, L_10));
	assert(liste_recherche(9, L_10));
	assert(!liste_recherche(62, L_10));

	ajouter_debut(L_10, 45);
	printf("Après avoir ajouté 45 au début:\n");
	liste_print(L_10);
	printf("\n");

	ajouter_fin(L_10, 31);
	printf("Après avoir ajouté 31 à la fin:\n");
	liste_print(L_10);
	printf("\n");

	supprimer_valeur(L_10, 7);
	printf("Après avoir supprimé 7:\n");
	liste_print(L_10);
	printf("\n");


	liste_free(L_10);
	return 0;
}