#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <assert.h>
#include <time.h>

/* Échange T[i] et T[j] */
void swap(int* T, int i, int j){
	int tmp = T[i];
	T[i] = T[j];
	T[j] = tmp;
}

/* Renvoie true si T est trié, false sinon.
   n est la taille de T */
bool is_sorted(int* T, int n){
	for (int i = 0; i < n-1; ++i){
		if(T[i] < T[i-1]){
			return false;
		}
	}
	return true;
}

/* Trie T, tableau de taille n */
void tri_insertion(int* T, int n){
	for (int i = 0; i < n; ++i){
		for (int j = i; j > 0 && T[j] < T[j-1]; j--){
			swap(T, j, j-1);
		}
	}
}


/** TRI RAPIDE DE BASE **/


/* Partitionne T[a..b] selon T[a] 
Renvoie un indice p tel que:
- T[p] contient l'ancienne valeur de T[a]
- Toutes les valeurs <= T[p] sont entre a et p-1
- Toutes les valeurs > T[p] sont entre p+1 et b
*/
int partition(int* T, int a, int b){
	assert(b-a+1>=2);
	int pivot = T[a];
	int i = a+1;
	int s = b;
	while(i<=s){
		if (T[i] > pivot){
			swap(T, i, s);
			s--;
		} else {
			i++;
		}
	}
	swap(T, i-1, a);
	return i-1;
}

/* Trie T[a..b] */
void tri_rapide_entre(int* T, int a, int b){
	if (b-a <= 0){
		return;
	}
	int p = partition(T, a, b);
	tri_rapide_entre(T, a, p-1);
	tri_rapide_entre(T, p+1, b);
}

/* Trie T, tableau de taille n */
void tri_rapide(int* T, int n){
	tri_rapide_entre(T, 0, n-1);
}


/** CHOIX DE PIVOT: MEDIANE DES TROIS **/
/*
Plutôt que de prendre systématiquement la première case comme
pivot, on regarde trois cases: la première, la dernière, et celle
du milieu. On choisit comme pivot la médiane des trois. 
*/

/* Renvoie l'indice de la médiane de T[a], T[b] et T[(a+b)/2*/
int mediane(int* T, int a, int b){
	int m = (a+b)/2;
	if (T[a] <= T[m] && T[m] <= T[b] || T[b] <= T[m] && T[m] <= T[a]){
		return m;
	} else 	if (T[m] <= T[a] && T[a] <= T[b] || T[b] <= T[a] && T[a] <= T[m]){
		return a;
	} else {
		return b;
	}
}


/* Partitionne T[a..b] selon la valeur médiane
   entre T[a], T[b] et T[m] où m est le milieu de [a,b] */
int partition_mediane(int* T, int a, int b){
	int med = mediane(T, a,b);
	// on met la médiane en T[a], puis on peut
	// réutiliser le même algorithme qu'avant
	swap(T, med, a);
	return partition(T, a, b);
}


/* Trie T[a..b] */
void tri_mediane_entre(int* T, int a, int b){
	if (b-a <= 0){
		return;
	}
	int p = partition_mediane(T, a, b);
	tri_mediane_entre(T, a, p-1);
	tri_mediane_entre(T, p+1, b);
}

/* Trie T, tableau de taille n */
void tri_mediane(int* T, int n){
	tri_mediane_entre(T, 0, n-1);
}



/** TRI HYBRIDE **/
int K = 10;
/* 
On arrête la procédure de tri rapide lorsque l'on atteint
des sous-tableaux de taille <= K
*/

/* Divise T[a..b] en blocs de tailles <= K, tels que
   chaque bloc contient globalement les bonnes valeurs 
   pour trier T[a..b], mais pas forcément dans l'ordre. 
   ex: T = [9, 1, 3, 4, 2, 1, 6, 4, 7] et K = 3:
   après hybrid_semi_sort, T pourra contenir:
   [1, 2, 1,    3, 4, 4,    9, 6, 7] */
void hybrid_semi_sort(int* T, int a, int b){
	if (b-a+1 <= K){
		return;
	} 
	int p = partition_mediane(T, a, b);
	hybrid_semi_sort(T, a, p-1);
	hybrid_semi_sort(T, p+1, b);	
}

/* Trie T, tableau de taille n */
void tri_hybride(int* T, int n){
	hybrid_semi_sort(T, 0, n-1);
	tri_insertion(T, n);
}



/** TESTS **/

/* Chaque fonction test_X_Y(int n, int nb_tests)
crée nb_tests tableaux de taille n en utilisant
la méthode de création X, et les trie
en utilisant l'algorithme de tri Y, puis renvoie
le temps moyen pris par tableaux. */


/* TESTS SUR DES TABLEAUX ALEATOIRES */
/* Initialise les n premières cases de T en y écrivant
 * des entiers aléatoires entre 0 et v_max inclus */
void random_tab(int* T, int n, int v_max){
	for (int i = 0; i < n; ++i){
		T[i] = rand()%(v_max+1);
	}
}

/* Trie nb_tests tableaux générés aléatoirement de taille n,
   en utilisant le tri rapide. Renvoie le temps moyen pris. */
float test_rand_rapide(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		random_tab(T, n, 2*n);
		tri_rapide(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_rand_mediane(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		random_tab(T, n, 2*n);
		tri_mediane(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_rand_insertion(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		random_tab(T, n, 2*n);
		tri_insertion(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_rand_hybride(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		random_tab(T, n, 2*n);
		tri_hybride(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

/* TESTS SUR DES TABLEAUX TRIES A L'ENVERS */
/* Initialise les n premières cases de T en y écrivant
   n-1, n-2, n-3, ..., 1, 0 
 */
void decroissant(int* T, int n){
	for (int i = 0; i < n; ++i){
		T[i] = n-i-1;
	}
}

float test_decr_rapide(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		decroissant(T, n);
		tri_rapide(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_decr_mediane(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		decroissant(T, n);
		tri_mediane(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_decr_insertion(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		decroissant(T, n);
		tri_insertion(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_decr_hybride(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		decroissant(T, n);
		tri_hybride(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}
/* TESTS SUR DES TABLEAUX QUASI-TRIES 
 */

/* Génère un tableau trié, puis échange n/10 paires de cases au hasard.
 */
void quasi_trie(int* T, int n){
	for (int i = 0; i < n; ++i){
		T[i] = i;
	}
	for (int k = 0; k < n/10; ++k){
		int i = rand()%n;
		int j = rand()%n;
		swap(T, i, j);
	}
}

float test_trie_rapide(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		quasi_trie(T, n);
		tri_rapide(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_trie_mediane(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		quasi_trie(T, n);
		tri_mediane(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_trie_insertion(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		quasi_trie(T, n);
		tri_insertion(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}

float test_trie_hybride(int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		quasi_trie(T, n);
		tri_hybride(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}


/* Teste les algorithmes de tri entre n = 100,000 et n = 1,000,000 */
int main(){
	srand(time(NULL));

	/* Génération des fichiers de valeurs */
	int nb_tests = 50;

	// Tableaux aléatoires
	FILE* f_rapide = fopen("valeurs/rand_rapide.txt", "w");
	FILE* f_insertion = fopen("valeurs/rand_insertion.txt", "w");
	FILE* f_mediane = fopen("valeurs/rand_mediane.txt", "w");


	/* Le format %-7.2f signifie "prends 7 caractères, et affiche avec 2 décimales".
	   Ca permet d'aligner joliment les valeurs */
	printf("\nTableaux aléatoires:\n");
	for (int n = 100; n <= 4000; n = n+100){
		float tr = test_rand_rapide(n, nb_tests);
		float ti = test_rand_insertion(n, nb_tests);
		float tm = test_rand_mediane(n, nb_tests);
		printf("n = %-5d:", n);
		printf(" Rapide: %7.2fms |", tr);
		printf(" Insertion: %7.2fms |", ti);
		printf(" Médiane: %7.2fms\n", tm);
		fprintf(f_rapide, "%f\n", tr);
		fprintf(f_insertion, "%f\n", ti);
		fprintf(f_mediane, "%f\n", tm);
	}
	fclose(f_rapide);
	fclose(f_insertion);
	fclose(f_mediane);

	// Tableaux décroissants
	f_rapide = fopen("valeurs/decr_rapide.txt", "w");
	f_insertion = fopen("valeurs/decr_insertion.txt", "w");
	f_mediane = fopen("valeurs/decr_mediane.txt", "w");
	printf("\nTableaux décroissants:\n");
	for (int n = 100; n <= 4000; n = n+100){
		float tr = test_decr_rapide(n, nb_tests);
		float ti = test_decr_insertion(n, nb_tests);
		float tm = test_decr_mediane(n, nb_tests);
		printf("n = %-5d:", n);
		printf(" Rapide: %7.2fms |", tr);
		printf(" Insertion: %7.2fms |", ti);
		printf(" Médiane: %7.2fms\n", tm);
		fprintf(f_rapide, "%f\n", tr);
		fprintf(f_insertion, "%f\n", ti);
		fprintf(f_mediane, "%f\n", tm);
	}
	fclose(f_rapide);
	fclose(f_insertion);
	fclose(f_mediane);

	// tris hybrides sur les tableaux quasi triés
	for (K = 5; K <= 500; K = K + 10){
		char nom_fichier[50];
		sprintf(nom_fichier, "valeurs/trie_hybride_%d.txt", K);

		FILE* f = fopen(nom_fichier, "w");
		for (int n = 10000; n <= 100000; n = n+10000){
			float th = test_trie_hybride(n, nb_tests);
			fprintf(f, "%f\n", th);
		}
		printf("Créé: %s\n", nom_fichier);
		fclose(f);
	}

	FILE* f = fopen("valeurs/trie_mediane.txt", "w");
		for (int n = 10000; n <= 100000; n = n+10000){
			float tm = test_trie_mediane(n, nb_tests);
			fprintf(f, "%f\n", tm);
		}
		printf("Créé: valeurs/trie_mediane.txt\n");
		fclose(f);	return 0;
}

