#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <assert.h>
#include <time.h>

void print_tab(int* T, int n){
	for (int i = 0; i < n; ++i){
		printf("%d ", T[i]);
	}
	printf("\n");
}

/* Échange T[i] et T[j] */
void swap(int* T, int i, int j){
	int tmp = T[i];
	T[i] = T[j];
	T[j] = tmp;
}

/* Renvoie true si T est trié, false sinon.
   n est la taille de T */
bool is_sorted(int* T, int n){
	for (int i = 0; i < n-1; ++i){
		if(T[i] < T[i-1]){
			return false;
		}
	}
	return true;
}

/* Trie T, tableau de taille n */
void tri_insertion(int* T, int n){
	for (int i = 0; i < n; ++i){
		for (int j = i; j > 0 && T[j] < T[j-1]; j--){
			swap(T, j, j-1);
		}
	}
}


/** TRI RAPIDE DE BASE **/


/* Partitionne T[a..b] selon T[a] 
Renvoie un indice p tel que:
- T[p] contient l'ancienne valeur de T[a]
- Toutes les valeurs <= T[p] sont entre a et p-1
- Toutes les valeurs > T[p] sont entre p+1 et b
*/
int partition(int* T, int a, int b){
	assert(b-a+1>=2);
	int pivot = T[a];
	int i = a+1;
	int s = b;
	while(i<=s){
		if (T[i] > pivot){
			swap(T, i, s);
			s--;
		} else {
			i++;
		}
	}
	swap(T, i-1, a);
	return i-1;
}

/* Trie T[a..b] */
void tri_rapide_entre(int* T, int a, int b){
	if (b-a <= 0){
		return;
	}
	int p = partition(T, a, b);
	tri_rapide_entre(T, a, p-1);
	tri_rapide_entre(T, p+1, b);
}

/* Trie T, tableau de taille n */
void tri_rapide(int* T, int n){
	tri_rapide_entre(T, 0, n-1);
}


/** CHOIX DE PIVOT: MEDIANE DES TROIS **/
/*
Plutôt que de prendre systématiquement la première case comme
pivot, on regarde trois cases: la première, la dernière, et celle
du milieu. On choisit comme pivot la médiane des trois. 
*/

/* Renvoie l'indice de la médiane de T[a], T[b] et T[(a+b)/2*/
int mediane(int* T, int a, int b){
	int m = (a+b)/2;
	if (T[a] <= T[m] && T[m] <= T[b] || T[b] <= T[m] && T[m] <= T[a]){
		return m;
	} else 	if (T[m] <= T[a] && T[a] <= T[b] || T[b] <= T[a] && T[a] <= T[m]){
		return a;
	} else {
		return b;
	}
 }

int partition_mediane(int* T, int a, int b){
	assert(b-a+1>=2);
	int med = mediane(T, a,b);
	// on met la médiane en T[a], puis on peut
	// réutiliser le même algorithme qu'avant
	swap(T, med, a);
	int pivot = T[a];
	int i = a+1;
	int s = b;
	while(i<=s){
		if (T[i] > pivot){
			swap(T, i, s);
			s--;
		} else {
			i++;
		}
	}
	swap(T, i-1, a);
	return i-1;
}

/* Trie T[a..b] */
void tri_mediane_entre(int* T, int a, int b){
	if (b-a <= 0){
		return;
	}
	int p = partition_mediane(T, a, b);
	tri_mediane_entre(T, a, p-1);
	tri_mediane_entre(T, p+1, b);
}

/* Trie T, tableau de taille n */
void tri_mediane(int* T, int n){
	tri_mediane_entre(T, 0, n-1);
}


/** TRI HYBRIDE **/
int K = 16;
/* 
On arrête la procédure de tri rapide lorsque l'on atteint
des sous-tableaux de taille <= K
*/

/* Divise T[a..b] en blocs de tailles <= K, tels que
   chaque bloc contient globalement les bonnes valeurs 
   pour trier T[a..b], mais pas forcément dans l'ordre. 
   ex: T = [9, 1, 3, 4, 2, 12, 1, 14, 6, 4, 7, 19] et K = 3:
   après hybrid_semi_sort, T pourra contenir:
   [1, 2, 1,    3, 4, 4,    9, 6, 7,   12, 14, 19] */
void hybrid_semi_sort(int* T, int a, int b){
	if (b-a+1 <= K){
		return;
	} 
	int p = partition_mediane(T, a, b);
	hybrid_semi_sort(T, a, p-1);
	hybrid_semi_sort(T, p+1, b);	
}

/* Trie T, tableau de taille n */
void tri_hybride(int* T, int n){
	hybrid_semi_sort(T, 0, n-1);
	tri_insertion(T, n);
}



/** TESTS **/

/* Trie nb_test tableaux de taille n générés par gen, 
	en utilisant l'algorithme tri. */
float test_tri(void gen(int*, int), void tri(int*, int), int n, int nb_tests){
	int* T = malloc(n*sizeof(int));
	clock_t deb = clock();
	for (int i = 0; i < nb_tests; ++i) {
		gen(T, n);
		tri(T, n);
		assert(is_sorted(T, n));
	}
	clock_t fin = clock();
	float total = (float) 1000.0f*(fin-deb)/CLOCKS_PER_SEC;
	return total / nb_tests;
}


/* TESTS SUR DES TABLEAUX ALEATOIRES
 * Initialise les n premières cases de T en y écrivant
 * des entiers aléatoires entre 0 et 2n inclus */
void random_tab(int* T, int n){
	for (int i = 0; i < n; ++i){
		T[i] = rand()%(2*n+1);
	}
}

/* TESTS SUR DES TABLEAUX TRIES A L'ENVERS
 * Initialise les n premières cases de T en y écrivant
   n-1, n-2, n-3, ..., 1, 0 
 */
void decroissant(int* T, int n){
	for (int i = 0; i < n; ++i){
		T[i] = n-i-1;
	}
}

/* TESTS SUR DES TABLEAUX QUASI-TRIES 
 *
 * Génère un tableau trié, puis échange n/10 paires de cases au hasard.
 */

void quasi_trie(int* T, int n){
	for (int i = 0; i < n; ++i){
		T[i] = i;
	}
	for (int k = 0; k < n/10; ++k){
		int i = rand()%n;
		int j = rand()%n;
		swap(T, i, j);
	}
}



/* Teste les algorithmes de tri entre n = 100,000 et n = 1,000,000 */
int main(int argc, char const *argv[]){
	srand(time(NULL));
	if (argc > 1){
		K = atoi(argv[1]);
	} else {
		printf("Seuil de taille pris pour le tri hybride: %d\n", K);
		printf("Pour modifier ce seuil, préciser la valeur en argument du programme.\n");
	}

	/* Le format %-7.2f signifie "prends 7 caractères, et affiche avec 2 décimales".
	   Ca permet d'aligner joliment les valeurs */
	int nb_tests = 10;

	/* Algos de tri */
	void (*tris[4]) (int*, int) = {
		tri_insertion, 
		tri_rapide, 
		tri_mediane, 
		tri_hybride
	};
	char* noms_tris[4] = {"Insertion", "Rapide", "Médiane", "Hybride"};

	/* Générateurs de tableaux */
	void (*gens[3]) (int*, int) = {random_tab, decroissant, quasi_trie};
	char* noms_gens[3] = {"Aléatoire", "Décroissant", "Quasi-trié"};


	for (int i = 0; i < 3; ++i){
		printf("Tableaux %s\n", noms_gens[i]);
		for (int n = 1000; n <= 10000; n=n+1000){
			printf("n = %d\n", n);
			for (int j = 0; j < 4; ++j){
				float t = test_tri(gens[i], tris[j], n, nb_tests);
				printf("%s: %7.2fms | ", noms_tris[j], t);
			}
			printf("\n");
		}	
	}

	return 0;
}

