from math import acos, sqrt, pi
import numpy as np
import matplotlib.pyplot as plt
import random


def gauche(p0, p1, p2):
    """
    Entrées: trois points p0, p1, p2
    Sortie: True si (p0, p1, p2) tourne à gauche, False sinon
    """
    x0, y0 = p0
    x1, y1 = p1
    x2, y2 = p2
    # signe du produit vectoriel
    return ((y1 -y2)*(x0-x1) - (x1-x2)*(y0-y1)) > 0

def graham(pts):
    """
    Entrée: pts une liste de points
    Sortie: liste des points de l'enveloppe convexe de pts
    """

    # trouver et enlever le point d'ordonnée minimale
    p0 = pts[0]
    for p in pts:
        if p[1] < p0[1]:
            p0 = p
        elif p[1] == p0[1] and p[0] > p0[0]:
            p0 = p
    pts.remove(p0)

    # trier les points par angle avec p0 et l'horizontale
    def angle(p):
        """
        renvoie l'angle de (p0, p) par rapport à la droite horizontale
        passant par p0.
        hypothèse: p0 est plus bas que p
        """
        x1, y1 = p0
        x2, y2 = p

        # cos = adjacent / hypothénuse
        cos_theta = (x2-x1) / sqrt((x2-x1)**2 + (y2-y1)**2)
        return acos(cos_theta)

    # on pouvait implémenter un des algos de tri vus en cours
    # comme le tri sélection ou le tri insertion, mais on peut
    # aussi utiliser la fonction sort pré-existante de python.
    # On peut donner en argument à cette fonction une fonction
    # utilisée pour la comparaison: en spécifiant key=angle, la
    # fonction va comparer deux points q et p selon les valeurs
    # de angle(p) et angle(q).
    pts.sort(key=angle)
    # une remarque: on aurait pu trier par cosinus décroissant plutôt que par
    # angle croissant, ce qui aurait donné le même résultat mais aurait permis
    # d'économiser le calcul de acos qui est coûteux.

    # itérer sur tous les points. On simule le passage pour p0
    # en le mettant directement sur la pile
    S = [p0] # pile des points
    for p in pts:
        while len(S)>1 and not gauche(S[-2], S[-1], p):
            S.pop()
        S.append(p)
    return S


def plot_pts(pts, link):
    """
    Entrées: pts une liste de points, link un booléen
    Sortie: Trace (sans afficher) les points de pts,
            en les reliant si link = True.
    """
    x = [p[0] for p in pts]
    y = [p[1] for p in pts]

    plt.plot(x, y, 'ro')
    if link:
        # Rajouter le premier point à la fin pour faire une boucle
        plt.plot(x+[x[0]], y+[y[0]], 'b-')



def test_graham(A=50, N=100):
    """
    génère environ N points aléatoires dans
    le cercle centré de rayon A, et affiche l'enveloppe convexe.
    A=50 et N=100 sont les paramètres par défaut, donc si l'on
    appelle simplement test_graham(), c'est équivalent à avoir appelé
    test_graham(50, 100)
    """
    pts = [(random.randint(-A, A), random.randint(-A, A)) for i in range(N)]
    # ne garder que les points dans le cercle
    pts = [(x, y) for (x, y) in pts if x*x+y*y < A*A]

    enveloppe = graham(pts)
    plot_pts(pts, False)
    plot_pts(enveloppe, True)
    plt.show()

# Q11
"""
La complexité de l'algorithme de Graham est en O(n log n).
La première étape consiste à trouver le point d'ordonnée minimale
(et d'abscisse maximale à égalité), en O(n)
La deuxième étape, consistant à trier la liste des points,
prend O(n log n) avec un tri efficace (c'est le cas du tri natif dans python)
La troisième étape est un parcours de la liste des points, aucours duquel
chaque point est empilé et dépilé au plus une fois sur la pile S. La complexité
de cette étape est donc en O(n).
Au total: O(n) + O(n log n) + O(n) = O(n log n)
"""

def estimer1_V(n):
    """
    Renvoie le nombre de points d'extrême d'un nuage de n points choisis
    uniforméments dans le carré unité.
    """
    pts = [(random.random(), random.random()) for i in range(n)]

    enveloppe = graham(pts)
    return len(enveloppe)

def estimer_V(n, nb_essais=50):
    """
    Renvoie la moyenne de `nb_essais` essais de estimer1_V(n)
    """

    s = 0
    for i in range(nb_essais):
        s += estimer1_V(n)
    return s / nb_essais

def plot_V():
    list_n = [5 * k for k in range(1, 400)]
    list_Vn = []
    for n in list_n:
        list_Vn.append(estimer_V(n))
        print(n)
    plt.plot(list_n, list_Vn, 'r-')
    plt.show()

# Q13
"""
A vue d'oeil Vn semble être proportionnelle à sqrt(n),
mais si on tente de tracer le rapport entre les deux,
on voit que ce n'est pas une constante:
"""
def test_sqrt():
    list_n = [2 * k for k in range(1, 200)]
    list_Vn = []
    for n in list_n:
        list_Vn.append(estimer_V(n, 30))
        print(n)

    # transformation en listes numpy
    list_n = np.array(list_n)
    list_Vn = np.array(list_Vn)

    plt.plot(list_n, list_Vn/(np.sqrt(list_n)))
    plt.show()

"""
On peut néanmoins imaginer que Vn est proportionnel à une puissance
de n entre 0 et 1/2. Testons cela en traçant Vn/(n**alpha) pour différentes
valeurs de alpha:
"""
def test_puissances():
    list_n = [100 * k for k in range(1, 100)]
    # calcul des valeurs de V(n)
    list_Vn = []
    for n in list_n:
        list_Vn.append(estimer_V(n, 30))
        print(n)

    list_n = np.array(list_n)
    list_Vn = np.array(list_Vn)

    # tracer le rapport entre V(n) et n**alpha
    for alpha in [0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 0.5]:
        plt.plot(list_n, list_Vn/(list_n**alpha), label=str(alpha))
    plt.legend()
    plt.show()

"""
Il n'y a pas de valeur qui semble coller de manière nette.
On peut essayer une nouvelle hypothèse: Vn est proportionnel
à ln(n)
"""
def test_log():
    list_n = [10 * k for k in range(10, 200)]
    list_Vn = []
    for n in list_n:
        list_Vn.append(estimer_V(n, 30))
        print(n)

    # transformation en listes numpy
    list_n = np.array(list_n)
    list_Vn = np.array(list_Vn)
    plt.plot(list_n, list_Vn/(np.log(list_n)))

    plt.show()

"""
Le test est beaucoup plus concluant: le rapport reste compris entre
2.4 et 2.8.
On peut en fait montrer que si l'on prend n points au hasard dans un
hypercube à d dimensions, le nombre de points extrême grandit
comme log(n)^(d-1). En particulier quand on est dans le point (d=2),
le nombre de points est bien proportionnel à log(n).
Un article sur le sujet: https://arxiv.org/pdf/1111.5340
"""