type expr =
	| Const of float (* constante *)
	| Add   of expr * expr (* Add(e1, e2) correspond à e1 + e2 *)
	| Mul   of expr * expr (* Mul(e1, e2) correspond à e1 * e2 *)
	| Var   of string
	| IFTE  of boolexpr * expr * expr

and boolexpr =
	| BConst of bool (* true et false *)
	| Or of boolexpr * boolexpr (* OU booléen *)
	| And of boolexpr * boolexpr (* ET booléen *)
	| Not of boolexpr (* NON booléen *)
	| Eq of expr * expr (* égalité *)
	| Leq of expr * expr (* inférieur ou égal *)

(* Exemple: représentation de (1 + 2) + (3 + (4 + 5)) *)
let e =
Add(
	Add(
		Const 1.,
		Const 2.
	),
	Add(
		Const 3.,
		Add(
			Const 4.,
			Const 5.
		)	
	)
)

type context = (string * float) list ;;

(* x -> 2.0, r -> -0.2 *)
let c1 = [("x", 2.0); ("r", -0.2)];;  

(* Nombre d'occurences de v dans e *)
let rec var_count (e: expr) (v: string) : int =
	match e with 
	| Const _ -> 0
	| Var x -> if x = v then 1 else 0
	| Add (e1, e2) | Mul (e1, e2) -> var_count e1 v + var_count e2 v
	| IFTE (b, e1, e2) -> var_count_bool b v + var_count e1 v + var_count e2 v

(* Nombre d'occurences de v dans b *)
and var_count_bool (b: boolexpr) (v: string) : int =
	match b with 
	| BConst _ -> 0
	| Not bb -> var_count_bool bb v
	| And (b1, b2) | Or (b1, b2) -> var_count_bool b1 v + var_count_bool b2 v
	| Eq (e1, e2) | Leq (e1, e2) -> var_count e1 v + var_count e2 v

(* valeur de e interprétée dans ctx *)
let rec eval (e: expr) (ctx: context) : float = 
	match e with 
	| Const x -> x 
	| Var v -> List.assoc v ctx
	| Add (e1, e2) -> 
		let v1 = eval e1 ctx in 
		let v2 = eval e2 ctx in 
		v1 +. v2
	| Mul (e1, e2) -> 
		let v1 = eval e1 ctx in 
		let v2 = eval e2 ctx in 
		v1 *. v2
	| IFTE (b, e1, e2) -> let vb = eval_bool b ctx in 
		if vb then eval e1 ctx
		else eval e2 ctx

(* valeur de b interprétée dans ctx *)
and eval_bool (b: boolexpr) (ctx: context) : bool =
	match b with 
	| BConst x -> x
	| Not bb -> not (eval_bool bb ctx)
	| And (b1, b2) -> 
		let v1 = eval_bool b1 ctx in 
		let v2 = eval_bool b2 ctx in 
		v1 && v2
	| Or (b1, b2) -> 
		let v1 = eval_bool b1 ctx in 
		let v2 = eval_bool b2 ctx in 
		v1 || v2
	| Eq (e1, e2) ->
		let e1 = eval e1 ctx in 
		let e2 = eval e2 ctx in 
		e1 = e2
	| Leq (e1, e2) ->
		let e1 = eval e1 ctx in 
		let e2 = eval e2 ctx in 
		e1 <= e2
